<?php

namespace App\Http\Controllers;

use App\Models\Department;
use App\Models\Doctor;
use App\Models\DoctorSeo;
use Illuminate\Http\Request;

class DoctorController extends Controller
{
    public function create(){
        $departments = Department::get();
        return view('backend.admin.doctor.create',compact('departments'));
    }

    public function store(Request $request) {
        $validatedData = $request->validate([
            'doctorfname' => 'required',
            'doctorlname' => 'required',
            'specialty' => 'required',
            'bio' => 'required',
            'phone' => ['nullable', 'regex:/^(?:(?:\+|0{0,2})91(\s*[-]\s*)?|[0]?)?[6789]\d{9}$/'],
            'email' => 'nullable|email',
            'gender' => 'required',
            'image' => 'required|image|mimes:jpeg,png,jpg,gif|max:2048',
            'department_id' => 'required|exists:departments,id',
        ]);
    
        $doctorData = [
            'doctorfname' => $validatedData['doctorfname'],
            'doctorlname' => $validatedData['doctorlname'],
            'specialty' => $validatedData['specialty'],
            'bio' => $validatedData['bio'],
            'gender' => $validatedData['gender'],
            'department_id' => $validatedData['department_id'],
        ];
    
        if (isset($validatedData['phone'])) {
            $doctorData['phone'] = $validatedData['phone'];
        }
    
        if (isset($validatedData['email'])) {
            $doctorData['email'] = $validatedData['email'];
        }
    
        if ($request->hasFile('image')) {
            $image = $request->file('image');
            $name = time() . '.' . $image->getClientOriginalExtension();
            $path = public_path('doctor/images');
            $image->move($path, $name);
            $doctorData['image'] = $name;
        }
    
        $doctor = new Doctor($doctorData);
        $doctor->save();
    
        return redirect()->back()->with('success', 'Doctor has been successfully added');
    }

    public function edit($id){
        $doctor = Doctor::findOrFail($id);
        $departments = Department::all();
        return view('backend.admin.doctor.edit', compact('doctor','departments'));
    }

    public function update(Request $request, $id)
    {
        $doctor = Doctor::findOrFail($id);
    
        $validatedData = $request->validate([
            'doctorfname' => 'required',
            'doctorlname' => 'required',
            'specialty' => 'required',
            'bio' => 'required',
            'phone' => ['nullable', 'regex:/^(?:(?:\+|0{0,2})91(\s*[-]\s*)?|[0]?)?[6789]\d{9}$/'],
            'email' => 'nullable|email',
            'gender' => 'required',
            'department_id' => 'required|exists:departments,id',
            'image' => 'image|mimes:jpeg,png,jpg,gif|max:2048|nullable', // Make the image field optional
        ]);
        
    
        $doctorData = [
            'doctorfname' => $validatedData['doctorfname'],
            'doctorlname' => $validatedData['doctorlname'],
            'specialty' => $validatedData['specialty'],
            'bio' => $validatedData['bio'],
            'gender' => $validatedData['gender'],
            'department_id' => $validatedData['department_id'],
        ];
    
        if (isset($validatedData['phone'])) {
            $doctorData['phone'] = $validatedData['phone'];
        }
    
        if (isset($validatedData['email'])) {
            $doctorData['email'] = $validatedData['email'];
        }
    
        if ($request->hasFile('image')) {
            // Handle the new image upload
            $image = $request->file('image');
            $name = time() . '.' . $image->getClientOriginalExtension();
            $path = public_path('doctor/images');
            $image->move($path, $name);
            $doctorData['image'] = $name;
        
            // Delete the old image if it exists
            if ($doctor->image) {
                $oldImagePath = public_path('doctor/images/' . $doctor->image);
                if (file_exists($oldImagePath)) {
                    unlink($oldImagePath);
                }
            }
        } elseif ($request->has('old_image')) {
            // Use the old image path if no new image is uploaded
            $doctorData['image'] = $request->input('old_image');
        }
        
    
        $doctor->update($doctorData);
    
        return redirect()->back()->with('success', 'Doctor has been successfully updated');
    }
    
    

    public function allDoctor(Request $request){
        $doctors = Doctor::get();
        return view('backend.admin.doctor.alldoctor',compact('doctors'));
    }

    public function show($id){
        
        $doctor = Doctor::findOrFail($id);
        if (!$doctor) {
            abort(404);
        }
        $departments = Department::get();
        $doctors = Doctor::get();
        $doctorMetaTitle = $doctor->doctorSeo;
         
        return view('frontend.pages.doctor.show',compact('doctor','departments','doctors','doctorMetaTitle'));
    }
    
    
    public function delete(Request $request , $id){
        $doctor = Doctor::findOrFail($id);
        $doctor->delete();
        return redirect()->route('doctor.all')->with('success','Doctor has been deleted successfully');
    }
    
    public function allDoctorForFronend(Request $request){
        $alldoctors = Doctor::get();
        return view('frontend.pages.doctor.alldoctors',compact('alldoctors'));
        
    }

}
  